/*
AD5662(16bit低速DAC)のSPIインターフェース
dacに16bitデータを入力
pulseの立ち上がりで、16bitデータをラッチして、
SPIに変換して出力する
*/
module DAC_SPI(clk,pulse,dac,sck,sdi,cs,dac_latch);

input clk;
input pulse;
input [15:0] dac;
output sck;
output sdi;
output cs;
output [15:0] dac_latch;

reg [15:0] shift_dac;
reg [24:0] shift_sdo_x;
reg shift_pulse;
reg [2:0] toggle;
reg [6:0] sdo_cnt;
reg cs;
reg sck;
reg [15:0] dac_latch;

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1)begin
		dac_latch <= shift_dac;
	end
	else begin
		dac_latch <= dac_latch; 		
	end
end

always@(posedge clk)begin
	shift_pulse <= pulse;
	shift_dac <= dac;
end

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1)
		toggle <= 0;
	else if(toggle==3)
		toggle <= 0;
	else
		toggle <= toggle+1'b1; 		
		
end

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1)
		sdo_cnt <= 0;
	else if(sdo_cnt != 30 && toggle ==3)
		sdo_cnt <= sdo_cnt + 1'b1; 	
	else
		sdo_cnt <= sdo_cnt;
end

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1)begin
		shift_sdo_x <= {8'h0,shift_dac};
	end
	else if(toggle==0)begin
		shift_sdo_x[24:1] <= shift_sdo_x[23:0] ;
		shift_sdo_x[0] <= 0; 
	end
	else begin
		shift_sdo_x <= shift_sdo_x; 		
	end
end

assign sdi = shift_sdo_x[24];

always @(posedge clk)begin
	if(sdo_cnt < 24)
		cs <= 0;
	else if(sdo_cnt > 25)
		cs <= 0;
	else
		cs <= 1;
end

always @(posedge clk)begin
	if(sdo_cnt < 24)
		sck <= ~toggle[1];
	else
		sck <= 1;
end




endmodule


/*
マイコンとのインターフェース（ＳＰＩ形式でのインターフェース）
*/
module SPI(cs,spck,sdi,sdo,m_clk,cnt1S,cnt40S,dac,jitter,reset,hour,minute,second);

input spck;
input sdi;
input m_clk;
input cs;
output sdo;
output reset;

output[5:0] hour;
output[5:0] minute;
output[5:0] second;




input [31:0] cnt1S;
input [31:0] cnt40S;
input [15:0] dac;
input [15:0] jitter;

wire fifo_out;
wire empty;
reg [23:0] shift_reg;   // Addr(8bit)+Data(16bit)    
reg [7:0] shift_cnt;

reg shift_empty;
reg shift_spck;
reg shift_shift_spck;

reg reset;

reg [16:0] shift_sdo;

MY_FIFO INST_FIFO1(
	.idata(sdi),
	.ock(m_clk),
	.ick(spck),
	.odata(fifo_out),
	.empty(empty));

	
always@(posedge m_clk)begin
	if(shift_empty==0)begin
		shift_reg[23:1] <= shift_reg[22:0];
		shift_reg[0] <= fifo_out;
	end
	else
		shift_reg <= shift_reg;		
end

always@(posedge m_clk)begin
	//if(shift_cs== 0 && cs==1)
	if(cs==0)
		shift_cnt <= 0;
	else if(cs==1 && shift_empty==0 && shift_cnt != 250)
		shift_cnt <= shift_cnt + 1'b1;
	else
		shift_cnt <= shift_cnt;		
end



always@(posedge m_clk)begin
	shift_empty <= empty;
	shift_spck <= spck;	
	shift_shift_spck <= shift_spck;		
//	shift_shift_shift_spck <= shift_shift_spck;
	//shift_cs <= cs;
end






/*
always@(posedge m_clk)begin
	if(shift_cnt == 23 && shift_empty==0 && shift_reg[21:15]==25 && shift_reg[22]==0)
		width3[19:16] <= {shift_reg[2:0],fifo_out};
	else
		width3[19:16] <= width3[19:16];		
end

*/


always@(posedge m_clk)begin
	if(shift_cnt == 23 && shift_empty==0 && shift_reg[21:15]==6 && shift_reg[22]==0)
		reset<= fifo_out;
	else
		reset <= reset;		
end
reg[5:0] hour;
reg[5:0] minute;
reg[5:0] second;
always@(posedge m_clk)begin
	if(shift_cnt == 23 && shift_empty==0 && shift_reg[21:15]==7 && shift_reg[22]==0)
		hour <=  {shift_reg[4:0],fifo_out};
	else
		hour <= hour;		
end

always@(posedge m_clk)begin
	if(shift_cnt == 23 && shift_empty==0 && shift_reg[21:15]==8 && shift_reg[22]==0)
		minute <=  {shift_reg[4:0],fifo_out};
	else
		minute <= minute;		
end

always@(posedge m_clk)begin
	if(shift_cnt == 23 && shift_empty==0 && shift_reg[21:15]==9 && shift_reg[22]==0)
		second <=  {shift_reg[4:0],fifo_out};
	else
		second <= second;		
end



always@(posedge m_clk)begin
	if(cs==0)
		shift_sdo<= 0;
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==0)
		shift_sdo <= cnt1S[15:0];		
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==1)
		shift_sdo <= cnt1S[31:16];	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==2)
		shift_sdo <= cnt40S[15:0];		
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==3)
		shift_sdo <= cnt40S[31:16];	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==4)
		shift_sdo <= dac;	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==5)
		shift_sdo <= jitter;	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==6)
		shift_sdo <= reset;	
		
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==7)
		shift_sdo <= hour;	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==8)
		shift_sdo <= minute;	
	else if(shift_cnt == 7 && shift_empty==0 && {shift_reg[5:0],fifo_out}==9)
		shift_sdo <= second;	
			
	else if(shift_spck==0 && shift_shift_spck==1) begin
		shift_sdo[16:1] <= shift_sdo[15:0];
		shift_sdo[0]   <= 0;
	end
	else
		shift_sdo<= shift_sdo;
end


assign sdo=shift_sdo[16];

endmodule




/*
上記SPIモジュールのサブモジュール
*/
module MY_FIFO(ick,ock, idata,odata,empty);

input ick;
input ock;
output odata;
input idata;
output empty;

//reg odata;
reg shift_ick;
reg shift_shift_ick;
//reg shift_shift_shift_ick;

assign empty =  ~((~ shift_shift_ick ) & shift_ick);

always@(posedge ock)begin
	shift_ick <= ick;
	shift_shift_ick <= 	shift_ick;
//	shift_shift_shift_ick <= 	shift_shift_ick;	
end

/*always@(posedge ock)begin
	if(empty==0)
		odata <= idata;	
	else
		odata <= odata;
end*/

assign odata=idata;

endmodule

/*
1PPSのチャタリング除去モジュール
1PPSにチャタリングがあると、
OCXO-PLL　および、1PPSジッター除去モジュールが誤動作するので
チャタリング除去する
*/
module Unti_Chatter(inpulse,m_clk,outpulse);
input inpulse;
input m_clk;
output outpulse;

reg outpulse;
reg [799:0] shift_reg;   //800bit = 10uS

always@(posedge m_clk)begin
	shift_reg[0] <= inpulse;
	shift_reg[799:1] <= shift_reg[798:0] ;	
end


always@(posedge m_clk)begin
	if(shift_reg)
		outpulse <= 1;
	else
		outpulse <= 0;	
end

endmodule


module delay(inpulse,m_clk,outpulse,div);
input inpulse;
input m_clk;
output outpulse;
output div;
reg [32:0] cnt;

assign div=cnt[14];

always@(posedge m_clk)begin
	cnt <= cnt + 1'b1;	
end


reg outpulse;
reg [799:0] shift_reg;   //800bit = 10uS

always@(posedge m_clk)begin
	shift_reg[0] <= inpulse;
	shift_reg[799:1] <= shift_reg[798:0] ;	
end


always@(posedge m_clk)begin
	outpulse <= shift_reg[200];	
end

endmodule


/*
OCXOと1PPSを同期させるためのPLL
OCXOを8逓倍した80MHzと、1PPSを比較する。
このモジュール内の分周器は、8千万分の1分周
このモジュールでロック判定も行います
pulseが、1PPSの入力です
*/
module PHASE_PLL(clk,pulse,dac,jitter,reset,lock,led_lock);
input clk;
input pulse;
output [15:0] dac;
output signed [15:0] jitter;
input reset;
output lock;
output led_lock;



reg signed [32:0] cap2 ;
reg shift_pulse;
reg signed [15:0] jitter;
reg signed [15:0] pre_jitter;
reg lock;
reg led_lock;
reg signed [40:0] cap1 ;
reg [34:0] div;
reg [37:0] init_cnt;
reg signed [16:0] pfd ;


always@(posedge clk)begin
	shift_pulse <= pulse;
end


always@(posedge clk)begin
	if(reset)
		init_cnt <= 0;
	else if(init_cnt < 80*1000*1000*60*45)
		init_cnt <= init_cnt + 1'b1;
	else
		init_cnt <= init_cnt;
end



//  10000分の1の分周動作のための
//　div 
always@(posedge clk)begin
	if(init_cnt < 90*1000*1000-1 && shift_pulse==0 && pulse==1)
		div <= 0;		
	else if(div==80*1000*1000-1)
		div <= 0;
	else
		div <= div + 1'b1;
end



//  周波数位相比較器のための
//　pfd 
always@(posedge clk)begin
	if(init_cnt < 95*1000*1000-1 && pulse==0)
		pfd  <= 0;		
	else if(pfd ==0 && div==0 && shift_pulse==0 && pulse==1)
		pfd  <= 0;
	else if(pfd ==0 && div==0)
		pfd  <= -1;
	else if(pfd ==0 &&  shift_pulse==0 && pulse==1)
		pfd  <= 1;
		
	else if(pfd ==1 && div==0)
		pfd  <= 0;
	else if(pfd ==-1 && shift_pulse==0 && pulse==1)
		pfd  <= 0;
	else
		pfd  <= pfd ;
end





//  ラグリードフィルタのC1に相当する
//　cap1
always@(posedge clk)begin
	if(init_cnt < 95*1000*1000-1)
		cap1  <= 0;
	else
		cap1  <= cap1  + pfd ;
end

//  ラグリードフィルタのR1とC2に相当する
//　cap2
always@(posedge clk)begin
	if(div==80*1000*1000-500)
		cap2  <= 0;
	else
		cap2  <= cap2 +pfd *8;
end

//  ラグリードフィルタのC1とC2とR1を全て合成する
assign dac = 32000+cap1 /16+cap2 ;


always@(posedge clk)begin
	if(div==80*1000*1000-500)
		pre_jitter <= 1000;
	else
		pre_jitter <= pre_jitter+pfd ;
end

always@(posedge clk)begin
	if(div==1000)
		jitter <= pre_jitter;
	else
		jitter <= jitter;
end


reg [11:0] ok_count_pll;
always@(posedge clk)begin
	if(init_cnt < 80*1000*1000*60*30)
	   ok_count_pll <= 0;
	else if(shift_pulse==0 &&  pulse==1  && 996 <= jitter && jitter <= 1004  && ok_count_pll<=2000)
	   ok_count_pll <= ok_count_pll + 1'b1;
	else if(shift_pulse==0 &&  pulse==1 )
	   ok_count_pll <= 0;
	 else
	   ok_count_pll <= ok_count_pll;	 
end


always@(posedge clk)begin
	if(ok_count_pll>=300)
	   lock <= 1;
   else
	   lock <= 0;	 
end


always@(posedge clk)begin
	if(ok_count_pll>=330)
	   led_lock <= 1;
	else if(init_cnt > 80*1000*1000*60*1 && init_cnt < 80*1000*1000*60*10 )
	   led_lock <= init_cnt[22];
	else if(init_cnt > 80*1000*1000*60*10 && init_cnt < 80*1000*1000*60*20 )
	   led_lock <= init_cnt[23];
	else if(init_cnt > 80*1000*1000*60*20 && init_cnt < 80*1000*1000*60*30 )
	   led_lock <= init_cnt[24];
	else if(ok_count_pll>=10)
	   led_lock <= pulse;
   else
	   led_lock <= 0;	 
end




endmodule

/*
GPS信号が途切れて、1PPSが途切れた場合のリセット信号を
生成する。
このモジュールの生成するリセット信号で、
OCXO-PLLをリセットする
*/
module GEN_RESET(clk,pulse,reset);
input clk;
input pulse;
output reset;

reg reset;
reg [31:0] cnt_1pps;
reg shift_pulse;
reg [7:0] ok_count_ocxo;
reg [11:0] ok_count_pll_p;

always@(posedge clk)begin
	shift_pulse <= pulse;
end

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1)
	   cnt_1pps <= 0;
	else if(cnt_1pps<=90*1000*1000)
	   cnt_1pps <= cnt_1pps + 1'b1;
   else
	   cnt_1pps <= cnt_1pps;	 
end

always@(posedge clk)begin
	if(shift_pulse==0 &&  pulse==1 && cnt_1pps > 80*1000*1000-50 && cnt_1pps < 80*1000*1000+50 && ok_count_ocxo<=5)
	   ok_count_ocxo <= ok_count_ocxo + 1'b1;
	else if(shift_pulse==0 &&  pulse==1 && cnt_1pps > 80*1000*1000-50 && cnt_1pps < 80*1000*1000+50)
	   ok_count_ocxo <= ok_count_ocxo;
	else if(shift_pulse==0 &&  pulse==1)
	   ok_count_ocxo <= 0;
	else if(cnt_1pps > 80*1000*1000+400)
	   ok_count_ocxo <= 0;
	 else
	   ok_count_ocxo <= ok_count_ocxo;	 
end



always@(posedge clk)begin
	if(ok_count_ocxo==6)
	   reset <= 0;
   else
	   reset <= 1;	 
end



endmodule

/*
FT2232Hと、FPGA内部モジュールを仲立ちするモジュール
*/
module FIFO_WRAPPER(ftdi_ck60m,ftdi_rxf,fdti_txf,ftdi_rd,ftdi_wr,ftdi_oe,ftdi_data,clk,rxf,txf,ren,wen,rdata,wdata);
input ftdi_ck60m;
input ftdi_rxf;
input fdti_txf;
output ftdi_rd;
output ftdi_wr;
output ftdi_oe;
input [7:0]ftdi_data;
input clk;
output rxf;
output txf;
input ren;
input wen;
output [7:0] rdata;
input [7:0] wdata;

reg [7:0] timer_oe;  //  1,2:idle 3 :pre_oe   4 :read  5:post_oe   6,7:idle      

wire [7:0] data_in;
wire [7:0] data_out;


reg tx_fifo_ren;
reg ftdi_wr;
wire tx_fifo_empty;
wire rx_fifo_full;

reg rx_fifo_wen;
reg ftdi_rd;
reg ftdi_oe;
reg [7:0] buf_oe;

always@(fdti_txf or tx_fifo_empty or timer_oe)begin
	if(fdti_txf==0 && tx_fifo_empty==0 && timer_oe==0)begin
		tx_fifo_ren <= 1;
		ftdi_wr <= 0;		
	end
	else begin
		tx_fifo_ren <= 0;
		ftdi_wr <= 1;		
	end
end



FIFO_8 INST_TX(
	.data(wdata),
	.rdclk(ftdi_ck60m),
	.rdreq(tx_fifo_ren),
	.wrclk(clk),
	.wrreq(wen),
	.q(data_out),
	.rdempty(tx_fifo_empty),
	.wrfull(txf));
	
always@(posedge ftdi_ck60m)begin
	if(ftdi_rxf==0 && timer_oe==0)
		timer_oe <= 1;
	else if(timer_oe==1)
		timer_oe <= 2;
	else if(timer_oe==2)
		timer_oe <= 3;
	else if(timer_oe==3)
		timer_oe <= 4;
	else if(timer_oe==4 && ftdi_rxf==1)
		timer_oe <= 5;
	else if(timer_oe==5)
		timer_oe <= 6;
	else if(timer_oe==6)
		timer_oe <= 7;
	else if(timer_oe==7)
		timer_oe <= 0;
	else 
		timer_oe <= timer_oe;
end



always@(ftdi_rxf or timer_oe or rx_fifo_full)begin
	if(ftdi_rxf==0 && timer_oe==4 && rx_fifo_full==0)begin
		rx_fifo_wen <= 1;
		ftdi_rd <= 0;		
	end
	else begin
		rx_fifo_wen <= 0;
		ftdi_rd <= 1;		
	end
end

always@(timer_oe)begin
	if(timer_oe==3 || timer_oe==4 || timer_oe==5 )begin
		ftdi_oe <= 0;		
	end
	else begin
		ftdi_oe <= 1;		
	end
end



always@(timer_oe)begin
	if(timer_oe==2 || timer_oe==3 || timer_oe==4 || timer_oe==5 || timer_oe==6 )begin
		buf_oe <= 0;		
	end
	else begin
		buf_oe <= 8'hFF;		
	end
end


IO_BUF_iobuf_bidir_a0p
	INST_IO( 
	.datain(data_out),
	.dataio(ftdi_data),
	.dataout(data_in),
	.oe(buf_oe)) ;

FIFO_8 INST_RX(
	.data(data_in),
	.rdclk(clk),
	.rdreq(ren),
	.wrclk(ftdi_ck60m),
	.wrreq(rx_fifo_wen),
	.q(rdata),
	.rdempty(rxf),
	.wrfull(rx_fifo_full));


endmodule


/*
デバッグ用
ADCの代わりのダミーデータを生成する
*/
module DUMMY_DATA(clk,pulse,cnt);
input clk;
input pulse;
output [13:0] cnt;

reg shift_pulse;
reg [13:0] cnt;

always@(posedge clk)begin
	shift_pulse <= pulse;
end

always@(posedge clk)begin
	if(shift_pulse==0 && pulse==1)
		cnt <= 0;
	else
		cnt <= cnt + 1'b1;	
end


endmodule

/*
ＰＣから、計測開始指令0xAA 計測終了指令0x55 を受け取る
モジュール
*/
module PC_Transfer_Enable(clk,enable,ren,rxf,rdata);
input clk;
output enable;
output ren;
input rxf;
input [7:0] rdata;

assign ren=~rxf;
reg enable;
reg shift_rxf;

always@(posedge clk)begin
	shift_rxf <= rxf;		
end


always@(posedge clk)begin
	if(shift_rxf==0 && rdata==8'hAA)
		enable <= 1;
	else if(shift_rxf==0 && rdata==8'h55)
		enable <= 0;
	else
		enable <= enable;		
end

endmodule


/*
ADCデータ転送、ＰＣへの各種ステータス報告など、
FPGA→PC転送の処理全般を受け持つ
*/
module TX_ASSHUKU(clk,adck,adc,wen,wdata,enable,txf,enable_25m,in_time,pulse_25m,latch_ennable_25m,lock_25m);

input clk;
input adck;
input [13:0] adc;
output wen;
output [7:0] wdata;
input enable;
input txf;
input enable_25m;
input [17:0] in_time;
input pulse_25m;
output latch_ennable_25m;
input lock_25m;

reg fifo_error;
reg pre_wen;
reg [7:0] wdata;
wire [17:0] q_adc;
wire fifo_full;
reg [13:0] previos_q_adc;
reg shift_enable;
wire rdempty;
wire fifo_error_80;
wire ocxo_error_80;

reg [7:0] state;
reg rdreq;
reg shift_rdreq;
reg shift_pulse_25m;
reg latch_ennable_25m;
reg [17:0] fifoin_25m;
reg shift_latch_ennable_25m;
reg unlock_alart_done;
reg unlock_alart;
reg [31:0] unlock_timer;


always@(posedge clk)begin
	shift_enable <= enable;	
end

always@(posedge clk)begin
	if(txf==0)
		shift_rdreq <= rdreq;
	else
		shift_rdreq <= shift_rdreq;		
end



always@(posedge adck)begin
	if(unlock_timer >= 25*1000*250)   // 250mS
		unlock_timer <= 0;
	else
		unlock_timer <= unlock_timer + 1'b1;		
end


always@(posedge adck)begin
	shift_pulse_25m <= pulse_25m;	
	shift_latch_ennable_25m <= latch_ennable_25m;
end

always@(posedge adck)begin
	if(shift_pulse_25m==0 && pulse_25m==1)
		latch_ennable_25m <= enable_25m;
	else
		latch_ennable_25m <= latch_ennable_25m;	
end

always@(posedge adck)begin
	if(latch_ennable_25m==0)
		fifo_error <= 0;
	else if(fifo_full==1)	
		fifo_error <= 1;	
	else if(fifo_full==0)	
		fifo_error <= 0;
	else
		fifo_error <= fifo_error;	
end

always@(posedge adck)begin
	if(unlock_timer==0)
		unlock_alart_done <= 0;
	else if(fifo_error==0 && lock_25m==0)	
		unlock_alart_done <= 1;	
	else
		unlock_alart_done <= unlock_alart_done;	
end

always@(unlock_alart_done or fifo_error or lock_25m)begin
	if(unlock_alart_done==1)
		unlock_alart <= 0;
	else if(fifo_error==0 && lock_25m==0)	
		unlock_alart <= 1;	
	else
		unlock_alart <= 0;	
end



always@(shift_latch_ennable_25m or latch_ennable_25m or in_time or adc)begin
	if(shift_latch_ennable_25m==0 && latch_ennable_25m==1 )
		fifoin_25m <= in_time;	
	else
		fifoin_25m <= adc;	
end


FIFO_20 INST_RX(
	.data({unlock_alart,fifo_error,fifoin_25m}),
	.rdclk(clk),
	.rdreq(rdreq),
	.wrclk(adck),
	.wrreq(~fifo_full && latch_ennable_25m), 
	.q({ocxo_error_80,fifo_error_80,q_adc}),
	.rdempty(rdempty),
	.wrfull(fifo_full));


always@(posedge clk)begin
	if(shift_enable==0 && enable==1)
		state <= 30;
	else if(txf==1)
		state <= state;
	else if(state==30 && rdempty==0)
		state <= 31;
	else if(state==0 && fifo_error_80 ==1 && shift_rdreq==1)
		state <= 21;
	else if(state==0 && ocxo_error_80 ==1 && shift_rdreq==1)
		state <= 41;
		
	else if(state==0 && previos_q_adc >= q_adc && previos_q_adc - q_adc < 120 && shift_rdreq==1)
		state <= 0;
	else if(state==0 && q_adc > previos_q_adc && q_adc - previos_q_adc < 120 && shift_rdreq==1)
		state <= 0;
	else if(state==0 && shift_rdreq==1)
		state <= 1;		
	else if(state==1)
		state <= 2;	
	else if(state==2)
		state <= 3;	
	else if(state==3)
		state <= 0;	

	else if(state==21)
		state <= 0;	
		
	else if(state==41)
		state <= 0;	

	else if(state==31)
		state <= 32;	
	else if(state==32)
		state <= 33;	
	else if(state==33)
		state <= 34;	
	else if(state==34)
		state <= 1;		
		
	else
		state <= state;	
end

always@(posedge clk)begin
	if(state==0 && previos_q_adc >= q_adc && previos_q_adc - q_adc < 120 && shift_rdreq==1)
		wdata <= 120-(previos_q_adc - q_adc);   // 120+(q_adc - previos_q_adc)
	else if(state==0 && q_adc > previos_q_adc && q_adc - previos_q_adc < 120 && shift_rdreq==1)
		wdata <= 120+(q_adc - previos_q_adc);
	else if(state==1)
		wdata <= 8'hFF;
	else if(state==2)
		wdata <= {2'h0,q_adc[13:8]};
	else if(state==3)
		wdata <= q_adc[7:0];
		
	else if(state==21)
		wdata <= 8'hFC;
		

	
	else if(state==41)
		wdata <= 8'hFA;
	
	else if(state==31)
		wdata <= 8'hFB;	
	else if(state==32)
		wdata <= q_adc[17:12];
	else if(state==33)
		wdata <= q_adc[11:6];
	else if(state==34)
		wdata <= q_adc[5:0];
		
		
	else
		wdata <= 0;		
end


always@(posedge clk)begin
	if(txf==1)
		previos_q_adc <= previos_q_adc;
	else if(state==0 && previos_q_adc >= q_adc && previos_q_adc - q_adc < 120 && shift_rdreq==1)
		previos_q_adc <=  q_adc;
	else if(state==0 && q_adc > previos_q_adc && q_adc - previos_q_adc < 120 && shift_rdreq==1)
		previos_q_adc <= q_adc;
	else if(state==3)
		previos_q_adc <= q_adc;
//	else if(state==22)
//		previos_q_adc <= q_adc;
	else
		previos_q_adc <= previos_q_adc;		
end

always@(posedge clk)begin
	if(state==0 && previos_q_adc >= q_adc && previos_q_adc - q_adc < 120 && shift_rdreq==1)
		pre_wen <= 1;
	else if(state==0 && q_adc > previos_q_adc && q_adc - previos_q_adc < 120 && shift_rdreq==1)
		pre_wen <= 1;
	else if(state==1)
		pre_wen <= 1;
	else if(state==2)
		pre_wen <= 1;
	else if(state==3)
		pre_wen <= 1;
		
	else if(state==21)
		pre_wen <= 1;
		
	else if(state==41)
		pre_wen <= 1;
		
	else if(state==31)
		pre_wen <= 1;
	else if(state==32)
		pre_wen <= 1;
	else if(state==33)
		pre_wen <= 1;
	else if(state==34)
		pre_wen <= 1;
		
	else
		pre_wen <= 0;		
end

assign wen=pre_wen & (~txf);

always@(state or rdempty or txf)begin
	if(txf==1)
		rdreq <= 0;
	else if(state==0 && rdempty==0)
		rdreq <= 1;
	else if(state==30 && rdempty==0)
		rdreq <= 1;
	else if(state==34 && rdempty==0)
		rdreq <= 1;	
		
	else
		rdreq <= 0;
end


endmodule


/*
1bitデータを異なる周波数ドメイン間でやり取りするモジュール
*/
module Frq_cnv1(inck,outck,indata,outdata);

input inck;
input outck;
input indata;
output outdata;

wire rdempty;
wire wrfull;

FIFO_1 INST_FIFO(
	.data(indata),
	.rdclk(outck),
	.rdreq(~rdempty),
	.wrclk(inck),
	.wrreq(~wrfull),
	.q(outdata),
	.rdempty(rdempty),
	.wrfull(wrfull));

endmodule

/*
1PPSのジッターを平均化により除去するモジュール
平均化された1PPSによってタイムスタンプを生成する
このモジュールは、ロックした後平均化処理をする
*/
module Pulse_average(clk,pulse,internal_pulse,lock);
input clk;
input pulse;

output internal_pulse;
input lock;

reg internal_pulse;
reg signed[31:0] internal_pulse_count;
reg signed[31:0] average;
reg [9:0] skip;

reg shift_pulse;
reg stp;
 

always@(posedge clk)begin
	shift_pulse <= pulse;
end

always@(posedge clk)begin
	if(shift_pulse==0 && pulse==1)
		stp <= 1;
	else
		stp <= 0;
end

always@(posedge clk)begin
	if(lock==0)
		skip <= 0;
	else if(shift_pulse==0 && pulse==1 && skip<500)
		skip <= skip + 1'b1;
	else
		skip <= skip;
end

always@(posedge clk)begin
	if(shift_pulse==0 && pulse==1 && skip < 10) 
		internal_pulse_count <= 0;
	else if(internal_pulse_count == 25*1000*1000-1) 
		internal_pulse_count <= 0;	
	else
		internal_pulse_count <= internal_pulse_count + 1'b1;	
end

always@(posedge clk)begin
	if(internal_pulse_count==average/16 && average>=0) 
		internal_pulse <= 1;	
	else if(internal_pulse_count==(25*1000*1000-1+average/16) && average<0) 
		internal_pulse <= 1;	
	else
		internal_pulse <= 0;	
end


always@(posedge clk)begin
	if(skip<6)
		average <= stp;
	else if(shift_pulse==0 && pulse==1 && skip<10)
		average <= 0;
	else if(shift_pulse==0 && pulse==1 && internal_pulse_count>20*1000*1000 && skip<26) 
		average <= average + ( internal_pulse_count - (25*1000*1000-1) );
	else if(shift_pulse==0 && pulse==1   && skip<26) 
		average <= average + internal_pulse_count;
	else
		average <= average;		
end




endmodule



/*
HH:MM:SS形式の時刻データを1秒進めるモジュール
GPS-UARTからマイコン経由でFPGAにUTCを転送すると
1秒遅れになるので、FPGA内部で1秒進める処理をする
*/
module Time_cnv1(inck,outck,in_h,in_m,in_s,outdata);

input inck;
input outck;
input [5:0] in_h;
input [5:0] in_m;
input [5:0] in_s;
output [17:0] outdata;

reg [5:0] temp_h;
reg [5:0] temp_m;
reg [5:0] temp_s;
wire rdempty;
wire wrfull;

always@(posedge inck)begin
	if(in_s == 59 ) 
		temp_s <= 0;	
	else
		temp_s <= in_s + 1'b1;	
end

always@(posedge inck)begin
	if(in_m == 59 && in_s==59) 
		temp_m <= 0;	
	else if(in_s==59)
		temp_m <= in_m + 1'b1;	
	else
		temp_m <= in_m;	
end

always@(posedge inck)begin
	if(in_h==23 && in_m == 59 && in_s==59) 
		temp_h <= 0;	
	else if(in_m == 59 && in_s==59) 
		temp_h <= in_h + 1'b1;	
	else
		temp_h <= in_h;	
end

FIFO_18 INST_FIFO(
	.data({temp_h,temp_m,temp_s}),
	.rdclk(outck),
	.rdreq(~rdempty),
	.wrclk(inck),
	.wrreq(~wrfull),
	.q(outdata),
	.rdempty(rdempty),
	.wrfull(wrfull));

endmodule


